/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package org.apache.directory.api.ldap.codec.actions.searchRequest;


import org.apache.directory.api.asn1.ber.grammar.GrammarAction;
import org.apache.directory.api.asn1.ber.tlv.TLV;
import org.apache.directory.api.ldap.codec.api.LdapMessageContainer;
import org.apache.directory.api.ldap.codec.decorators.SearchRequestDecorator;
import org.apache.directory.api.ldap.model.message.SearchRequest;
import org.apache.directory.api.ldap.model.message.SearchRequestImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * The action used to initialize the SearchRequest message
 * <pre>
 * LdapMessage ::= ... SearchRequest ...
 * SearchRequest ::= [APPLICATION 3] SEQUENCE { ...
 * </pre>
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class InitSearchRequest extends GrammarAction<LdapMessageContainer<SearchRequestDecorator>>
{
    /** The logger */
    private static final Logger LOG = LoggerFactory.getLogger( InitSearchRequest.class );


    /**
     * Instantiates a new action.
     */
    public InitSearchRequest()
    {
        super( "Init SearchRequest" );
    }


    /**
     * {@inheritDoc}
     */
    public void action( LdapMessageContainer<SearchRequestDecorator> container )
    {
        // Now, we can allocate the SearchRequest Object
        TLV tlv = container.getCurrentTLV();

        SearchRequest internalSearchRequest = new SearchRequestImpl();
        internalSearchRequest.setMessageId( container.getMessageId() );
        SearchRequestDecorator searchRequest = new SearchRequestDecorator(
            container.getLdapCodecService(), internalSearchRequest );

        searchRequest.setTlvId( tlv.getId() );
        container.setMessage( searchRequest );

        LOG.debug( "Search Request" );
    }
}
