#if   !defined(__CORELINUXGUARDPOOL_HPP)
#define __CORELINUXGUARDPOOL_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__SINGLETON_HPP)
#include <Singleton.hpp>
#endif

#if   !defined(__SEMAPHOREGROUP_HPP)
#include <SemaphoreGroup.hpp>
#endif

#if   !defined(__VECTOR_HPP)
#include <Vector.hpp>
#endif

#if   !defined(__MAP_HPP)
#include <Map.hpp>
#endif


namespace corelinux
{


   /**
   PoolDescriptor stores information about
   the ownership and usage of a semaphore
   in the pool
   */

   struct   PoolDescriptor
   {
      /// The number of Synchronized concurrent request

      Count    theQueueLength;

      /// The SemaphoreGroup that this semaphore belongs to.

      Index    theGroupIndex;
   };

   CORELINUX_VECTOR
      (
       SemaphoreGroupPtr,
       GroupVector
      );

   CORELINUX_MAP
      ( 
         AbstractSemaphorePtr, 
         PoolDescriptor, 
         std::less<AbstractSemaphorePtr>,
	 SemaphoreMap
      );

   CORELINUX_MAP
      ( 
         SynchronizedPtr,
         AbstractSemaphorePtr,
         std::less<SynchronizedPtr>,
	 MonitorMap
      );

   DECLARE_CLASS( CoreLinuxGuardPool );
   DECLARE_TYPE( Singleton<CoreLinuxGuardPool>, GuardPool );

   /**
   The CoreLinuxGuardPool is a private SemaphoreGroup resource for class 
   objects that require synchronization capability.
   @see corelinux::Synchronized, corelinux::PoolDescriptor
   */

   class CoreLinuxGuardPool
   {
   public:

      //
      // Constructors and destructor
      //

                        /**
                        Default constructor
                        @param Short number of base semaphores to use in pool
                        @param Short number of semaphores to increment by
                        when going into extents. If 0, no extents are allowed.
                        @exception Assertion if Singleton<CoreLinuxGuardPool>
                        already established.
                        */

                        CoreLinuxGuardPool( Short numInit=8, Short numExt=0 )
                           throw( Assertion );

                        /// Virtual destructor

      virtual           ~CoreLinuxGuardPool( void );

      //
      // Operator overloads
      //

      //
      // Accessors
      //

               /**
               isLocked determines if the object is currently locked.
               Calls singleton instance isSynchronizedLocked.
               @param Synchronized pointer of guard owner.
               @return bool true if locked
               */

      static   bool  isLocked( SynchronizedPtr ) throw( SemaphoreException );

               /// Return the initial guard count in the pool

      static   Short getInitialPoolSize( void );

               /// Return the current grow by count for the pool

      static   Short getExtentSize( void );

               /// Return the current guard pool size

      static   Short getTotalCurrentSize( void );

      //
      // Mutators
      //

               /**
               lock is called by a guard when control is needed over
               a objects resource access. Calls singleton instance
               lockedSynchronized.
               @param Synchronized pointer of guard owner.
               @exception SemaphoreException if out of pool resources
               */

      static   void  lock( SynchronizedPtr ) throw( SemaphoreException );

               /**
               release is called by a guard object during its destruction.
               @param Synchronized pointer of guard owner.Calls singleton 
               instance releaseSynchronized.
               @exception SemaphoreException if Synchronized object held
               no resource.
               */
               
      static   void  release( SynchronizedPtr ) throw( SemaphoreException );

               /**
               Run time interface for changing the extent size. The next
               time the pool goes into extent processing, this will be
               used.
               @param short the new extent size
               @exception Assertion if size < 0
               */

      static   void  setExtentSize( Short aExtentSize ) throw( Assertion );

   protected:

      //
      // Accessors
      //
               
               /**
               isSynchronizedLocked resolves whether Synchronized
               is in a locked state.
               @param Synchronized pointer of guard owner.
               @return bool true if locked
               @exception SemaphoreException if not of set.
               */

               bool  isSynchronizedLocked( SynchronizedPtr ) 
                           throw( SemaphoreException );

      //
      // Mutators
      //

               /**
               lockSynchronized manages the associations of objects
               to the semaphore in the pool in establishing the
               guard.
               @param Synchronized pointer of guard owner.
               */

               void  lockSynchronized( SynchronizedPtr ) 
                           throw( SemaphoreException );

               /**
               releaseSynchronized manages the associations of objects
               to the semaphore in the pool when releasing a guard.
               @param Synchronized pointer of guard owner.
               */

               void  releaseSynchronized( SynchronizedPtr ) 
                           throw( SemaphoreException );

               /**
               createPoolGroup creates a semaphore set with the
               requested number of semaphores in the group and
               will add the semaphores to theSemaphores with
               initial count and index set properly
               @param   Short the semaphore count
               @param   Short the number to insert into theSemaphores
               If the default is used, all go into theSemaphores
               */

               void  createPoolGroup( Short numSems, Short initSize=0 );

               /**
               destroyPoolGroup validates that all the semaphores in
               the extent are not being used and then destroys the
               extent and all the semaphores associated with it. The
               method assumes that the group is the last in the
               vector.
               @param Index the group index
               */

               void  destroyPoolGroup( Index aGroup );

   protected:

               /// Singleton instance

      static   GuardPool            theGuard;

               /// Describes the inital pool size

      static   Short                theInitialSize;

               /// Describes the size to add when going into extents

      static   Short                theExtentSize;

   private:

               /// Vector of base and extents

               GroupVector          theGroups;

               /// Map of semaphores to reference counts

               SemaphoreMap         theSemaphores;

               /// Map of Synchronized objects to semaphores

               MonitorMap           theCallers;

               /// A internal mutex semaphore

               AbstractSemaphorePtr theControlSem;
   };
}

#endif // if !defined(__CORELINUXGUARDPOOL_HPP)

/*
   Common rcs information do not modify
   $Author: prudhomm $
   $Revision: 1.3 $
   $Date: 2000/08/31 22:52:20 $
   $Locker:  $
*/

