/*
 * (C) Copyright 2002, Schlund+Partner AG
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// Local configuration
#include "config.h"

// Implementation
#include "XMLNodeSet.hpp"

// STDC++
#include <cassert>

// Local
#include "XMLTree.hpp"

namespace SP {
namespace GXML {

// Copy of XPathContext from XMLTree.cpp
class XPathContextNS
{
public:
	XPathContextNS(xmlDocPtr const doc)
		:_context(xmlXPathNewContext(doc))
	{
		if (!_context)
		{
			throw XMLException(XMLException::XMLTREE_CREATE_CONTEXT);
		}
	}

	~XPathContextNS()
	{
		xmlXPathFreeContext(_context);
	}

	xmlXPathContextPtr get() const
	{
		return _context;
	}

private:
	xmlXPathContextPtr const _context;
};

XMLNodeSet::XMLNodeSet( const std::string& xpath, const XMLTree* tree )
{
	XPathContextNS context(tree->getDocPtr());

	xPathObj_ = xmlXPathEval( (xmlChar*)xpath.c_str(), context.get());

	// this seem never to happen...
	if ( xPathObj_ == NULL ) {
		throw XMLException( XMLException::XMLNODESET_EVAL );
	}
	assert( xPathObj_ );

	// happens if xpath doesn't point to node(s)
	if ( xPathObj_->nodesetval == NULL ) {
		throw XMLException( XMLException::XMLNODESET_NO_SET );
	}
	assert( xPathObj_->nodesetval );

	// happens if xpath doesn't exist
	if ( xPathObj_->nodesetval->nodeNr <= 0 ) {
		throw XMLException( XMLException::XMLNODESET_NO_MATCH );
	}
	assert( xPathObj_->nodesetval->nodeNr > 0 );
}

XMLNodeSet::~XMLNodeSet()
{
	xmlXPathFreeObject( xPathObj_ );
}

int
XMLNodeSet::size()
{
	assert( xPathObj_ && xPathObj_->nodesetval );
	return xPathObj_->nodesetval->nodeNr;
}

XMLNodeSet::Iterator
XMLNodeSet::begin()
{
	assert( xPathObj_ && xPathObj_->nodesetval );
	return XMLNodeSet::Iterator( xPathObj_->nodesetval );
}

XMLNodeSet::Iterator
XMLNodeSet::end()
{
	return XMLNodeSet::Iterator();
}

////////////////////////////////////////////////////////////////////////////////
// NodeSet Iterator
//
XMLNodeSet::Iterator::Iterator()
{
	pos_ = 0;
	set_ = NULL;
}

XMLNodeSet::Iterator::Iterator( const xmlNodeSetPtr set )
{
	pos_ = 0;
	set_ = set;
}

XMLNodeSet::Iterator::~Iterator()
{}

XMLNodeSet::Iterator
XMLNodeSet::Iterator::operator ++()
{
	assert( set_ );
	if ( pos_ < set_->nodeNr - 1 ) {
		++pos_;
	} else {
		pos_ = 0;
		set_ = NULL;
	}
	return *this;
}

bool
XMLNodeSet::Iterator::operator !=( const XMLNodeSet::Iterator& iter )
{
	if ( ( this->pos_ != iter.pos_ ) || ( this->set_ != iter.set_ ) ) {
		return true;
	} else {
		return false;
	}
}

XMLNode
XMLNodeSet::Iterator::operator *()
{
	assert( set_ && set_->nodeTab[pos_] );
	XMLNode result( set_->nodeTab[pos_] );
	// return copy of XMLNode
	return result;
}

int
XMLNodeSet::Iterator::getPosition()
{
	assert( set_ );
	return pos_;
}

}}
