/* install.c -- MiKTeX Installation Utility
   Time-stamp: "97/08/29 09:34:45 cschenk"

   Copyright (C) 1996, 97 Christian Schenk <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <conio.h>
#include <string.h>
#include <ctype.h>
#include <process.h>

#include "global.h"
#include "md5.h"

//#ifndef _M_IX86
#define NO_UNZIP_DIGEST
//#endif

#include "../lib/miktex.version"

#define PROGRAM_NAME (VER_PRODUCTNAME_STR " " VER_PRODUCTVERSION_STR)
#define DEF_DEST_DIR "c:\\texmf"

#if defined (BETA)
#define HOWTOGET ""
#else
#define HOWTOGET "\
You can download missing files from this ftp location:\n\
\n\
   ftp://ftp.dante.de/tex-archive/systems/win32/miktex"
#endif

#define default_mode		required
#define required		1
#define optional		2
#define ask			4
#define dontcare		8
#define ignore			16
#define ask_if_other_tds	32

#define isslash(c) ((c) == '/' || (c) == '\\')

static int
lastch (const char *s)

{
  size_t len;

  len= strlen (s);
  return (len == 0 ? 0 : s[len - 1]);
}

static char destdir[_MAX_PATH];
static char destdirsansbackslash[_MAX_PATH];
static char tdspath[_MAX_PATH * 10];
static char root_directories[_MAX_PATH * 11];
static char old_miktex_bin_path[_MAX_PATH];

#define other_tds_p() (tdspath[0] != 0)

static int
yesno (const char *	question,
			...)

{
  va_list marker;

  va_start (marker, question);
  do
    {
      char answer[100];
      vfprintf (stdout, question, marker);
      fflush (stdout);
      gets (answer);
      if (_strcmpi (answer, "yes") == 0)
	{
	  va_end (marker);
	  return (1);
	}
      else if (_strcmpi (answer, "no") == 0)
	{
	  va_end (marker);
	  return (0);
	}
      else
	{
	  fprintf (stdout, "`%s'?; please enter either `yes' or `no'!\n",
		   answer);
	}
    }
  while (1);
}

static int
continue_anyway ()

{
  return (yesno ("Do you whish to continue anyway [yes/no]? "));
}

static struct fileinfo
{
  const char *	filename;
  int		mode;
  const char *	description;
//  unsigned char	digest[16];
}

filelist[] =

{
#include "install.dat"
};

static void
inslog (const char *	msg,
			...)

{
  va_list marker;
  va_start (marker, msg);
  vfprintf (stdout, msg, marker);
  va_end (marker);
  fputc ('\n', stdout);
}

static void
error (const char *	msg,
			...)

{
  va_list marker;
  fprintf (stderr, "Error: ");
  va_start (marker, msg);
  vfprintf (stderr, msg, marker);
  va_end (marker);
  fputc ('\n', stderr);
  fprintf (stderr, "Press any key to continue...\n");
  getch ();
  exit (1);
}

static void
warning (const char *	msg,
			...)

{
  va_list marker;
  fprintf (stderr, "Warning: ");
  va_start (marker, msg);
  vfprintf (stderr, msg, marker);
  va_end (marker);
  fputc ('\n', stderr);
}

static void
usage (int rc)

{
  fprintf (stderr, "Usage: install\n");
  exit (rc);
}

static int
get_file_digest (const char *	filename,
		 unsigned char	digest[16])

{
  FILE *file;

  file = fopen (filename, "rb");

  if (file == 0)
    return (-1);
  else
    {
      unsigned char buffer[4096];
      MD5_CTX context;
      size_t len;

      MD5Init (&context);
      
      while (len = fread (buffer, 1, sizeof (buffer), file))
	MD5Update (&context, buffer, len);

      MD5Final (digest, &context);
    }

 fclose (file);

 return (0);
}

#ifndef NO_UNZIP_DIGEST
/* The md5 digest of unzip.exe (as it is distributed together with
   MiKTeX). */
static unsigned char
unzip_digest[16] =

{
  0x4a, 0xe1, 0xe2, 0x56,
  0x8d, 0x99, 0x9a, 0x33,
  0x46, 0x9a, 0x48, 0x6c,
  0x89, 0xde, 0xca, 0x33,
};
#endif

static void
check_unzip_utility ()

{
#ifndef NO_UNZIP_DIGEST
  unsigned char digest[16];

  /* Unzip.exe must be there. */
  if (_access (".\\unzip.exe", 0) != 0)
    error ("UNZIP.EXE not found in current directory!");

  /* Get the digest of unzip.exe. */
  get_file_digest (".\\unzip.exe", digest);

  /* Compare the digests. */
  if (memcmp (&digest, &unzip_digest, sizeof (digest)) != 0)
    warning ("Hmm, found an UNZIP.EXE of unknown origin!");
#else
  if (_access (".\\unzip.exe", 0) != 0)
    error ("UNZIP.EXE not found in current directory!");
#endif
}

static void
make_path (const char *path)

/* borrowed from GNU File Utilities! */

{
#define S_ISDIR(m) (((m) & _S_IFMT) == _S_IFDIR)
  char *dirpath;
  struct _stat stats;
  int retval = 0;
  dirpath = (char *) _alloca (strlen (path) + 1);
  strcpy (dirpath, path);
  if (_stat (dirpath, &stats) != 0)
    {
      char *slash;
      char *cp;
      struct ptr_list
      {
	char *dirname_end;
	struct ptr_list *next;
      };
      struct ptr_list *p, *leading_dirs = NULL;
      slash = dirpath;
      if (slash[0] && slash[1] == ':')
	slash += 2;
      else if (isslash (slash[0]) && slash[1] == slash[0])
	{
	  /* UNC file name. */
	  slash += 2;
	  while (*slash && ! isslash (*slash))
	    slash++;
	  if (isslash (*slash))
	    {
	      slash++;
	      while (*slash && ! isslash (*slash))
		slash++;
	    }
	}
      while (isslash (*slash))
	slash++;
      while ((cp = strchr (slash, '/')) || (cp = strchr (slash, '\\')))
	{
	  slash = cp;
	  *slash = '\0';
	  if (_stat (dirpath, &stats) != 0)
	    {
	      if (_mkdir (dirpath) != 0)
		{
		  error ("cannot make directory '%s'", dirpath);
		}
	    }
	  else if (! S_ISDIR (stats.st_mode))
	    {
	      error ("'%s' exists but is not a directory", dirpath);
	    }
	  *slash++ = '\\';
	  while (*slash == '/' || *slash == '\\')
	    slash++;
	}
      if (_mkdir (dirpath) != 0)
	{
	  error ("cannot make directory '%s'", dirpath);
	}
    }
  else
    {
      if (! S_ISDIR (stats.st_mode))
	{
	  error ("'%s' exists but is not a directory", dirpath);
	}
    }
}

static void
check_dest_dir (void)

{
  make_path (destdir);
}

static void
uncompress (size_t idx)

{
  int pargc;
  int rc;
  const char *pargv[10];
  pargc = 0;  
  pargv[pargc++] = "unzip.exe";
  pargv[pargc++] = "-qq";
  pargv[pargc++] = "-o";
  pargv[pargc++] = "-d";
  pargv[pargc++] = destdir;
  pargv[pargc++] = filelist[idx].filename;
  pargv[pargc++] = 0;
  inslog ("Installing %s...", filelist[idx].description);
  rc = _spawnv (_P_WAIT, ".\\unzip.exe", pargv);
  if (rc < 0)
    perror (".\\unzip.exe");
  if (rc != 0)
    error ("could not unpack archive `%s'", filelist[idx].filename);
}

static int
check_file_digest (size_t i)

{
  unsigned char digest[16];
  get_file_digest (filelist[i].filename, digest);
  return (memcmp (filelist[i].digest, digest, sizeof (digest)) == 0);
}

static void
process_filelist (void)

{
#define nfiles (sizeof (filelist) / sizeof (struct fileinfo))

  size_t i;
  int missing;
  int tampered;
  int exitflag;
  int first_question;

  missing = 0;
  exitflag = 0;
  tampered = 0;
  first_question = 1;

  inslog ("Checking integrity of archives...\n");

  for (i = 0; i < nfiles; i++)
    {
      if (_access (filelist[i].filename, 0) != 0)
	{
	  if ((filelist[i].mode & required))
	    {
	      inslog ("Error: required archive `%s' does not exist",
		      filelist[i].filename);
	      exitflag++;
	    }
	  else if ((filelist[i].mode & optional))
	    {
	      inslog ("Warning: archive `%s' does not exist",
		      filelist[i].filename);
	      missing++;
	    }
	}
      else if (! check_file_digest (i))
	{
	  inslog ("%s: this archive has been tampered with",
		  filelist[i].filename);
	  tampered++;
	}
      else if (((filelist[i].mode & ask)
		|| ((filelist[i].mode & ask_if_other_tds) && other_tds_p ()))
	       && ! exitflag)
	{
	  if (first_question)
	    {
	      inslog ("\n\nSelect optional components that you wish \
to install:\n");
	      first_question = 0;
	    }
	  if (! yesno ("\n%s ? [yes/no] ", filelist[i].description))
	    filelist[i].mode |= ignore;	  
	}
    }

  inslog ("\n\n");

  if (exitflag)
    error ("%d archive%s %s missing\n\n%s",
	   exitflag, exitflag > 1 ? "s" : "",
	   exitflag > 1 ? "are" : "is",
	   HOWTOGET);

  if (missing)
    {
      inslog ("%d archive%s %s missing",
	      missing, missing > 1 ? "s" : "",
	      missing > 1 ? "are" : "is");
      if (! continue_anyway ())
	{
	  inslog ("  %s", HOWTOGET);
	  exit (0);
	}
    }

  if (tampered)
    {
      inslog ("%d archive%s %s been tampered with",
	      tampered, tampered > 1 ? "s" : "",
	      tampered > 1 ? "have" : "has");
      if (! continue_anyway ())
	{
	  inslog ("  %s", HOWTOGET);
	  exit (0);
	}
    }

  check_dest_dir ();

  inslog ("\n");
  for (i = 0; i < nfiles; i++)
    {
      if (_access (filelist[i].filename, 0) != 0)
	{
	  if (filelist[i].mode & required)
	    error ("required archive `%s' (%s) does not exist",
		   filelist[i].filename, filelist[i].description);
	}
      else if (! (filelist[i].mode & ignore))
	uncompress (i);
    }
}

#define CONFIGURE_EXE "configure.exe"

static void
configure_and_check ()

{
  FILE *stream;
  char command_line[300];
  char system_temp_dir[_MAX_PATH];
  char cwd[_MAX_PATH];
  int drive;
  int temp_drive;
  int pargc;
  char *pargv[10];
  int rc;
  char *oldpath;
  char *newpath;
  int old_bin_p;
  
  old_bin_p = 0;
  if (*old_miktex_bin_path)
    {
      char new_miktex_bin_path[_MAX_PATH];
      sprintf (new_miktex_bin_path, "%s\\miktex\\bin", destdirsansbackslash);
      if (_strcmpi (new_miktex_bin_path, old_miktex_bin_path) != 0)
	old_bin_p = 1;
    }

  oldpath = getenv ("PATH");
  if (oldpath == 0)
    error ("environment variable PATH is not defined");
  newpath = (char *) _alloca (strlen (oldpath)
			      + strlen ("PATH=")
			      + strlen (destdirsansbackslash)
			      + strlen ("\\miktex\\bin") + 10);
  sprintf (newpath, "PATH=%s\\miktex\\bin;%s",
	   destdirsansbackslash, oldpath);
  if (putenv (newpath) != 0)
    error ("could not modify PATH");

  sprintf (command_line, "%s\\miktex\\config\\%s",
	   destdirsansbackslash, CONFIGURE_EXE);

  inslog ("\nWriting file name data base...\n");

  pargc = 0;
  pargv[pargc++] = command_line;
  pargv[pargc++] = "-r";
  pargv[pargc++] = root_directories;
  pargv[pargc++] = "-u";
  pargv[pargc++] = 0;
  
  rc = _spawnv (_P_WAIT, command_line, pargv);
  if (rc < 0)
    perror (command_line);
  if (rc != 0)
    error ("Configuration Utility failed");
  GetTempPath (sizeof (system_temp_dir), system_temp_dir);
  drive = _getdrive ();
  temp_drive = tolower (system_temp_dir[0]) - 'a' + 1;
  if (getcwd (cwd, sizeof (cwd)) == 0)
    {
      perror ("");
      error ("cannot determine current directory");
    }
  if (_chdrive (temp_drive) != 0)
    {
      perror (system_temp_dir);
      error ("cannot chdrive");
    }
  if (_chdir (system_temp_dir) != 0)
    {
      perror (system_temp_dir);
      error ("cannont chdir");
    }
  if ((stream = fopen ("test.tex", "w")))
    {
      size_t i;
      fprintf (stream, "\\hsize=10cm\\parindent=0pt\\raggedright%%\n\
Installation of MiK\\TeX\\ %s is complete.\n\n\
Do not forget to append {\\tt ", VER_PRODUCTVERSION_STR);
      for (i = 0; i < strlen (destdirsansbackslash); i++)
	{
	  if (destdirsansbackslash[i] == '\\')
	    fprintf (stream, "$\\backslash$");
	  else
	    fputc (destdirsansbackslash[i], stream);
	}
      fprintf (stream,"$\\backslash$miktex$\\backslash$bin} to the \
environment variable {\\tt PATH}.\n\n\
Send problem reports to cschenk@berlin.snafu.de.\n\n\
Enjoy!\n\
\\bye",
	       destdirsansbackslash);
      fclose (stream);
      inslog ("\n\nChecking that the installation worked:\n\n");
      inslog ("  1. Invoking TeX...\n");
      sprintf (command_line, "%s\\miktex\\bin\\tex.exe",
	       destdirsansbackslash);
      pargc = 0;
      pargv[pargc++] = command_line;
      pargv[pargc++] = "test";
      pargv[pargc++] = 0;
      rc = _spawnv (_P_WAIT, command_line, pargv);
      if (rc < 0)
	perror (command_line);
      if (rc)
	error ("Invocation of TeX failed");
      inslog ("\n  2. Opening DVI file...\n");
      sprintf (command_line, "%s\\miktex\\bin\\yap.exe",
	       destdirsansbackslash);
      pargc = 0;
      pargv[pargc++] = command_line;
      pargv[pargc++] = "test.dvi";
      pargv[pargc++] = 0;
      rc = _spawnv (_P_WAIT, command_line, pargv);
      if (rc < 0)
	perror (command_line);
      if (rc)
	error ("Invocation of YAP failed");
    }
  if (_chdrive (drive) != 0)
    perror (cwd);
  if (chdir (cwd) != 0)
    perror (cwd);

  if (old_bin_p)
    {
      inslog ("Your PATH variable points to an old MiKTeX bin directory!");
    }
}

static void
welcome (void)

{
  inslog ("Welcome to %s Installation Utility. This program will install\n\
%s on your system.\n\n", PROGRAM_NAME, PROGRAM_NAME);
}

static char *
find_dest_dir (char *		dest_dir,
	       const char *	def_dest_dir)

{
  HKEY hKey;
  LONG res;
  char *cp;

  res = RegOpenKeyEx (HKEY_LOCAL_MACHINE,
		      "Software\\MiK\\MiKTeX\\CurrentVersion\\MiKTeX",
		      0,
		      KEY_READ,
		      &hKey);
  if (res != ERROR_SUCCESS)
    res = RegOpenKeyEx (HKEY_LOCAL_MACHINE,
			"Software\\MiK\\MiKTeX\\MiKTeX",
			0,
			KEY_READ,
			&hKey);
  if (res == ERROR_SUCCESS)
    {
      DWORD type;
      BYTE value[2048];
      DWORD len = sizeof(value);
      res = RegQueryValueEx (hKey,
			     "TEXMF Root Directories",
			     0,
			     &type,
			     value,
			     &len);
      if (res == ERROR_SUCCESS)
	{
	  char *cp = strchr (value, ';');
	  if (cp)
	    *cp = 0;
	}
      else
	res = RegQueryValueEx (hKey,
			       "Root Directory",
			       0,
			       &type,
			       value,
			       &len);
      RegCloseKey (hKey);
      if (res == ERROR_SUCCESS)
	strcpy (dest_dir, value);
      else
	strcpy (dest_dir, def_dest_dir);
    }
  else
    strcpy (dest_dir, def_dest_dir);
  cp = strchr (dest_dir, ';');
  if (cp)
    *cp = 0;
  return (0);
}

static void
get_dest_dir (void)

{
  size_t i;
  size_t len;
  char def_dest_dir[300];
  find_dest_dir (def_dest_dir, DEF_DEST_DIR);
  inslog ("%s will be installed in `%s'.\n", PROGRAM_NAME, def_dest_dir);
  inslog ("To install to this directory, press <RETURN>.\n");
  inslog ("To install to a different directory, enter the name of another \
directory.\n");
  inslog ("You can choose not to install %s. Enter `cancel' to cancel the\n\
installation.\n", PROGRAM_NAME);
  printf ("Destination Directory: ");
  gets (destdir);
  if (*destdir == 0)
    strcpy (destdir, def_dest_dir);
  if (strcmp (destdir, "cancel") == 0)
    exit (0);

  len = strlen (destdir);
  for (i = 0; i < len; i++)
    {
      if (destdir[i] == '/')
	destdir[i] = '\\';
    }
  strcpy (destdirsansbackslash, destdir);
  if (destdirsansbackslash[len - 1] == '\\')
    destdirsansbackslash[len - 1] = 0;
  if (! ((destdir[0] == '\\' && destdir[1] == '\\')
	 || (isalpha (destdir[0]) && destdir[1] == ':' && destdir[2] == '\\')))
    error ("you must enter an absolute pathname (e.g. c:\\texmf)");
}

static void
get_tds_path (void)

{
  size_t i;
  size_t len;
  inslog ("\nYou can cause MiKTeX to incorporate additional TEXMF trees.");
  inslog ("To do so, enter a semicolon-separated list of directory names.");
  inslog ("Press <RETURN>, to use nothing but the TEXMF tree that comes with MiKTeX.\n");
  printf ("Incorporate other TEXMF trees: ");
  gets (tdspath);
  putchar ('\n');
  len = strlen (tdspath);
  for (i = 0; i < len; i++)
    {
      if (tdspath[i] == '/')
	tdspath[i] = '\\';
    }
}

static int
access_directory (const char *	filename,
		  int		mode)

{
  char myfilename[_MAX_PATH];
  DWORD attr;
  strcpy (myfilename, filename);
  if (! isslash (lastch (myfilename)))
    strcat (myfilename, "\\");
  attr = GetFileAttributes (myfilename);
  return (attr == 0xffffffff
	  ? -1
	  : (attr & FILE_ATTRIBUTE_DIRECTORY
	     ? 0
	     : -1));
}

static int
check_tds_path (const char *roots)

{
  char *my_roots = strdup (roots);
  int ok = 1;
  char *root = strtok (my_roots, ";");
  while (root && *root)
    {
      if (access_directory (root, 0) != 0)
	{
	  warning ("%s: this directory does not exist", root);
	  ok = 0;
	}
      else
	{
	  char subdir[_MAX_PATH];
	  _makepath (subdir, 0, root, "tex", 0);
	  if (access_directory (subdir, 0) != 0)
	    {
	      warning ("%s: this doesn't seem to be a standard TEXMF \
root directory",
		       root);
	      ok = 0;
	    }
	}
      root = strtok (0, ";");
    }

  free (my_roots);

  if (! ok && ! continue_anyway ())
    exit (1);
   
  return (ok);
}

static void
locate_old_miktex_bin ()

{
  char *filename;
  if (SearchPath (0, "miktex.dll", 0,
		  sizeof (old_miktex_bin_path),
		  old_miktex_bin_path, &filename) == 0)
    *old_miktex_bin_path = 0;
  else
    {
      if (filename > old_miktex_bin_path)
	filename--;
      *filename = 0;
    }
}

int
main (int argc,
      char **argv)

{
  if (argc != 1)
    usage (-1);

  welcome ();

  get_dest_dir ();
  get_tds_path ();
  check_tds_path (tdspath);

  strcpy (root_directories, destdir);
  strcat (root_directories, ";");
  strcat (root_directories, tdspath);

  locate_old_miktex_bin ();
  check_unzip_utility ();
  process_filelist ();
  configure_and_check ();
  inslog ("\n\nInstallation is complete. Please append `%s\\miktex\\bin' to\n\
the environment variable `PATH' and reboot the system.",
       destdirsansbackslash);
  return (0);
}

/* install.c ends here */
